/***********************************************************************************************************
*
***********************************************************************************************************/
#ifndef CFXLIBRARY_COMPILE
#include "FXCCompile.hlsl"
#endif

/***********************************************************************************************************
*
***********************************************************************************************************/
struct VS_INPUT
{
    float4 Pos          : POSITION;
	float3 Normal		: NORMAL;
	float3 Tangent		: TANGENT;
	float3 Binormal		: BINORMAL;
    float2 TexCoord     : TEXCOORD0;
};

/***********************************************************************************************************
*
***********************************************************************************************************/
#include "Uniforms.hlsl"
#include "Transforms.hlsl"
#include "Lighting.hlsl"
#include "Textures.hlsl"
#include "Shadow.hlsl"
#include "PhongTools.hlsl"
#include "Debug.hlsl"

/***********************************************************************************************************
*
***********************************************************************************************************/
float2 TexCoordRatio : ATTRIBUTE = float2(1.0f, 1.0f);

/***********************************************************************************************************
*
***********************************************************************************************************/
texture BaseMap     < string NTM = "Base"; >; 
texture GlossMap    < string NTM = "Gloss"; >;
texture LightMap    < string NTM = "Shader"; int NTMIndex = 0; >;
texture BaseMap2    < string NTM = "Shader"; int NTMIndex = 1; >;
texture MaskMap     < string NTM = "Shader"; int NTMIndex = 2; >;
texture NormalMap   < string NTM = "Shader"; int NTMIndex = 3; >;
texture FresnelMap  < string NTM = "Shader"; int NTMIndex = 4; >;

/***********************************************************************************************************
*
***********************************************************************************************************/
sampler BaseSampler     = sampler_state { Texture = (BaseMap); };
sampler GlossSampler    = sampler_state { Texture = (GlossMap); };
sampler LightSampler    = sampler_state { Texture = (LightMap); };
sampler Base2Sampler    = sampler_state { Texture = (BaseMap2); };
sampler MaskSampler     = sampler_state { Texture = (MaskMap); };
sampler NormalSampler   = sampler_state { Texture = (NormalMap); };
sampler FresnelSampler  = sampler_state { Texture = (FresnelMap); ADDRESSU = CLAMP; ADDRESSV = CLAMP; };

/***********************************************************************************************************
*
***********************************************************************************************************/
struct VS_OUTPUT
{
    float4 Pos               : POSITION;
	float3 Normal			 : NORMAL;
	float3 Tangent			 : TANGENT;
	float3 Binormal			 : BINORMAL;
	float4 WorldPos          : TEXCOORD2;
    float3 TexCoord0         : TEXCOORD0;
	float3 TexCoord1         : TEXCOORD1;
};

/***********************************************************************************************************
*
***********************************************************************************************************/
VS_OUTPUT VSMain(VS_INPUT In)
{
    VS_OUTPUT Out = (VS_OUTPUT)0;

	Out.TexCoord0.xy = GetUV(In.TexCoord.xy, TexTransformBase, HasBaseMapTransform);
    Out.TexCoord1.xy = GetUV(In.TexCoord.xy, TexTransformBase, HasBaseMapTransform) * TexCoordRatio;

    float4x3 WorldTransform = GetWorldTransform(In);
    float3 vWorldPos = mul(In.Pos, WorldTransform);
    float4x4 WorldViewProjTransform = GetWorldViewProjTransform(In);
    Out.Pos = mul(In.Pos, WorldViewProjTransform);
	
	Out.WorldPos.xyz = vWorldPos;
	
	Out.Normal = normalize(mul(In.Normal, (float3x3)WorldTransform));
    Out.Tangent = normalize(mul(In.Tangent, (float3x3)WorldTransform));
    Out.Binormal = normalize(mul(In.Binormal, (float3x3)WorldTransform));

    return Out;
}

/***********************************************************************************************************
*
***********************************************************************************************************/
void ComputeNormal(out float3 vOutNormal, 
                       float2 vTexCoord,
                       float3 vInTangent, 
                       float3 vInBinormal, 
                       float3 vInNormal)
{
	float4 vNormalMap = tex2D(NormalSampler, vTexCoord, HasShaderMap3);
	if(AGBNormalMap)
	{
		vNormalMap.rgba = vNormalMap.agbr;
	}
	float3 vTS_Normal = 255.0 / 128.0 * vNormalMap.rgb - 1;
	vOutNormal = vTS_Normal.r * vInTangent + vTS_Normal.g * vInBinormal + vTS_Normal.b * vInNormal;

    vOutNormal = normalize(vOutNormal);
}

/***********************************************************************************************************
*
***********************************************************************************************************/
void ComputeFresnel(out float kFresnelCoeff, out float3 kFresnelLighting, float NdotV)
{
    kFresnelCoeff = tex1D(FresnelSampler, 1.0-NdotV, HasShaderMap4).r;
    kFresnelLighting = kFresnelCoeff;
}

/***********************************************************************************************************
*
***********************************************************************************************************/
#define PS_OUTPUT PSOutput
PS_OUTPUT PSMain(VS_OUTPUT In)
{
	PS_OUTPUT sOut = (PS_OUTPUT)1;
    float4 vBaseSampler   = tex2D(BaseSampler, In.TexCoord1, HasBaseMap);
	float4 vGlossSampler  = tex2D(GlossSampler, In.TexCoord0, HasGlossMap);
	float4 vLightSampler  = tex2D(LightSampler, In.TexCoord0, HasShaderMap0);
	float4 vBase2Sampler  = tex2D(Base2Sampler, In.TexCoord0, HasShaderMap1);
	float4 vMaskSampler   = tex2D(MaskSampler, In.TexCoord0, HasShaderMap2);
	
	float3 vNormal;
    ComputeNormal(vNormal, In.TexCoord0, In.Tangent, In.Binormal, In.Normal);
	
	float3 vFromEye = normalize(In.WorldPos - EyePos);
	float4 vFresnelLighting = float4(1.0f,1.0f,1.0f,1.0f);
    float fFresnelCoeff = 0.0f;
    ComputeFresnel(fFresnelCoeff, vFresnelLighting.xyz, abs(dot(-vFromEye, vNormal)));
	
	sOut.vColor0 = lerp(vBase2Sampler, vBaseSampler, vMaskSampler.r) * vLightSampler + (vGlossSampler + vFresnelLighting);
	
	return sOut;
}

/***********************************************************************************************************
*
***********************************************************************************************************/
technique Shirt
<
string ShaderName           = "Shirt";
string classname            = "SHMaterial";
string NBTMethod            = "ATI";
int NBTSourceUV             = 0;
bool UsesNiRenderState      = true;
bool UsesNiSamplerState     = true;
>
{
    pass P0
    {
        VertexShader    = compile vs_3_0 VSMain();
        PixelShader     = compile ps_3_0 PSMain();
    }
}
